
// SimParm: Simple and flexible C++ configuration framework
// Copyright (C) 2007 Australian National University
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// Contact:
// Kevin Pulo
// kevin.pulo@anu.edu.au
// Leonard Huxley Bldg 56
// Australian National University, ACT, 0200, Australia

#include "ConfigSet.hh"
#include <fstream>
#include <sys/select.h>

ConfigSet::ConfigSet()
{
	attached.setName("attached");
	attached.setDesc("Attached");
	attached.setViewable(false);
	attached.setEditable(false);
	attached.setOutputOnChange(false);
	attached = false;
	register_entry(&attached);

	//attached = false;

	setIO(NULL, -1, NULL, -1);
}

ConfigSet::ConfigSet(istream *in, int infd, ostream *out, int outfd)
{
	ConfigSet();
	setIO(in, infd, out, outfd);
}

void ConfigSet::setIO(istream *in, int infd, ostream *out, int outfd) {
	this->in = in;
	this->infd = infd;
	this->out = out;
	this->outfd = outfd;
	if (in) {
		in->sync_with_stdio(true);
	}
	if (out) {
		out->sync_with_stdio(true);
		out->rdbuf()->pubsetbuf(NULL, 0);
		// this is now deferred until attach time
		//output_definition(*out);
	}
}

void ConfigSet::pulse() {
	// FIXME: only output pulse if we are sure there is something there
	// listening for it (and not the user's tty or the user's pbs
	// job logfile)
	if (out && attached) {
		(*out) << "pulse" << endl;
		out->flush();
	}
}

bool ConfigSet::inputWaiting() {
	// this is the bodgy c++ way of doing it.
	// it no longer works in g++ 3.4.
	//return in->rdbuf()->in_avail();

	// the C select way.
	// at least this works.
	fd_set readfds;
	FD_ZERO(&readfds);
	//int fd = fileno(stdin);
	//int fd = in->rdbuf()->fileno();
	int fd = infd;
	FD_SET(fd, &readfds);
	struct timeval tmout;
	tmout.tv_sec = 0;
	tmout.tv_usec = 0;
	select(fd + 1, &readfds, NULL, NULL, &tmout);
	return FD_ISSET(fd, &readfds);
}


void ConfigSet::serviceInput() {
	if ( ! in) {
		return;
	}

	// ARGHHHHHHHHHHHH
	if ( ! attached) {
		return;
	}

	if ( ! inputWaiting()) {
		//cerr << "no input waiting" << endl;
		pulse();
		return;
	}

	if (in->eof()) {
		// FIXME return false so that this fin can be removed
		//config.finished.setValue(true);
		cerr << "EOF received on control channel" << endl;
	} else {
		string cmd;
		while (inputWaiting()) {
			(*in) >> cmd;
			if ( ! cmd.compare("set") ) {
				//cerr << "GOT A SET COMMAND!!" << endl;
				inputSingleEntry(*in);
			} else if ( ! cmd.compare("pulse") ) {
				// ignore heartbeats
			} else if ( ! cmd.compare("attach") ) {
				attached = true;
				cerr << "Attached" << endl;
				if (out) {
					output_definition(*out);
				}
			} else if ( ! cmd.compare("request") ) {
				string what;
				(*in) >> what;
				if ( ! what.compare("definition") ) {
					if (out) {
						output_definition(*out);
					}
				} else {
					cerr << "Unknown request: \"" << what << "\"" << endl;
				}
			} else {
				cerr << "Unknown command: \"" << cmd << "\"" << endl;
			}
		}
		pulse();
	}
}

void ConfigSet::register_entry(ConfigEntry *entry) {
	if (entry) {
		//cerr << "registering " << entry->name() << " with " << entry->configSet() << endl;
		if (entry->configSet() == NULL) {
			entry->setConfigSet(this);
		}
		register_unowned_entry(entry);
	}
}

void ConfigSet::register_unowned_entry(ConfigEntry *entry) {
	if (entry) {
		lookup[entry->name()] = entry;
		ordered.push_back(entry);
	}
}

void ConfigSet::output_definition(ostream &out) {
	out << ordered.size() << endl;
	for (vector<ConfigEntry*>::const_iterator entry = ordered.begin(); entry != ordered.end(); entry++) {
		(*entry)->output_definition(out);
	}
	out << endl;
	out.flush();
}

void ConfigSet::addChangeCallback(changecallback_t callback) {
	changecallbacks.push_back(callback);
}

void ConfigSet::removeChangeCallback(changecallback_t callback) {
	// FIXME
	changecallbacks.erase(find(changecallbacks.begin(), changecallbacks.end(), callback));
}

ConfigEntry* ConfigSet::operator[](string name) {
	return lookup[name];
}

istream& ConfigSet::inputSingleEntry(istream& in) {
	string name;
	string dummy;

	if (in >> name) {
		if (name[0] == '#') {
			char ch;
			while (in.peek() != '\n' && in.peek() != '\r') {
				in.get(ch);
			}
			while (in.peek() == '\n' || in.peek() == '\r') {
				in.get(ch);
			}

		} else {

			in >> dummy;
			if (dummy != "=") {
				cerr << "Error reading config entry: expected '=', found \"" << dummy << "\"" << endl;
				return in;
			}

			if (lookup.find(name) == lookup.end()) {
				cerr << "Error reading config entry: unknown parameter name \"" << name << "\"" << endl;
				return in;
			}

			ConfigEntry *entry = lookup[name];
			in >> *entry;
			if (entry->invalid()) {
				cerr << "Error reading config entry: error reading value for parameter \"" << name << "\"" << endl;
				return in;
			}

			cerr << "Set: " << *entry;
		}
	}
	return in;
}

istream& ConfigSet::inputAllEntries(istream& in) {
	string name;
	string dummy;
	unsigned long linenum = 0;

	while (in >> name) {
		linenum++;

		if (name[0] == '#') {
			char ch;
			while (in.peek() != '\n' && in.peek() != '\r') {
				in.get(ch);
			}
			while (in.peek() == '\n' || in.peek() == '\r') {
				in.get(ch);
			}
			continue;
		}

		in >> dummy;
		if (dummy != "=") {
			cerr << "Error reading config file on line " << linenum << ": expected '=', found \"" << dummy << "\"" << endl;
			break;
		}

		if (lookup.find(name) == lookup.end()) {
			cerr << "Error reading config file on line " << linenum << ": unknown parameter name \"" << name << "\"" << endl;
			break;
		}

		ConfigEntry *entry = lookup[name];
		in >> *entry;
		if (entry->invalid()) {
			cerr << "Error reading config file on line " << linenum << ": error reading value for parameter \"" << name << "\"" << endl;
			break;
		}

	}
	cerr << "Read " << linenum << " lines from config file" << endl;

	return in;
}

bool ConfigSet::readConfig(int argc, char *argv[]) {

	for (int i = 1; i < argc; i++) {
		char *config_fname;
		ifstream config_fin;

		// FIXME: detect all of these:
		// --<option>=<value>
		// --"<option> = <value>"
		// --<option> <value>
		// --<option> = <value>
		// <option> = <value>
		// --<booloption>
		// --no<booloption>
		// --no-<booloption>
		// --enable-<booloption>
		// --disable-<booloption>
		// --set-<booloption>
		// --unset-<booloption>

		config_fname = argv[i];
		cerr << "Reading config file " << config_fname << endl;
		config_fin.open(config_fname);
		config_fin >> (*this);
		if ( ! config_fin.eof()) {
			cerr << "Error reading config file" << endl;
			return 1;
		}
		config_fin.close();
	}

	return 0;
}

istream& operator>>(istream& in, ConfigSet &configSet) {
	return configSet.inputAllEntries(in);
}

ostream& operator<<(ostream& out, const ConfigSet &configSet) {
	for (vector<ConfigEntry*>::const_iterator entry = configSet.ordered.begin(); entry != configSet.ordered.end(); entry++) {
		out << **entry;
	}
	return out;
}


