
// SimParm: Simple and flexible C++ configuration framework
// Copyright (C) 2007 Australian National University
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// Contact:
// Kevin Pulo
// kevin.pulo@anu.edu.au
// Leonard Huxley Bldg 56
// Australian National University, ACT, 0200, Australia

#if !defined(CONFIGENTRY_HH)
#define CONFIGENTRY_HH

#include <iostream>
#include <string>
#include <vector>

using namespace std;

class ConfigSet;
class ConfigEntry;

typedef void (*changecallback_t)(const ConfigEntry*);

class ConfigEntry {
	protected:
		string _name;
		string _desc;
		bool _invalid;
		bool _viewable;
		bool _editable;
		bool _outputOnChange;
		ConfigSet *_configSet;
		vector<changecallback_t> changecallbacks;

	public:
		ConfigEntry();
		ConfigEntry(ConfigSet *configSet);
		ConfigEntry(string name, string desc);
		ConfigEntry(string name, string desc, ConfigSet *configSet);
		ConfigEntry(const ConfigEntry &entry);
		virtual ~ConfigEntry();

		void setName(const string &name);
		void setDesc(const string &desc);
		void setInvalid(const bool &invalid);
		void setViewable(const bool &viewable);
		void setEditable(const bool &editable);
		void setOutputOnChange(const bool &outputOnChange);
		void setConfigSet(ConfigSet *configSet);
		void addChangeCallback(changecallback_t callback);
		void removeChangeCallback(changecallback_t callback);

		const string& name() const;
		const string& desc() const;
		const bool& invalid() const;
		const bool& viewable() const;
		const bool& editable() const;
		const bool& outputOnChange() const;
		const ConfigSet* configSet() const;

		void notifyChangeCallbacks() const;
		void outputIfNecessary() const;

		ConfigEntry &operator=(const ConfigEntry &entry);

		void output_definition(ostream &out) const;
		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		virtual void output() const;
		friend istream& operator>>(istream& in, ConfigEntry &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntry &entry);
};


class ConfigEntryString : public ConfigEntry {
	protected:
		string _value;

	public:
		ConfigEntryString();
		ConfigEntryString(const ConfigEntryString &entry);
		ConfigEntryString(string name, string desc);
		ConfigEntryString(string name, string desc, const string &value);
		ConfigEntryString(string name, string desc, const char *value);

		const string &value() const;
		const string &operator()() const;
		void setValue(const string &value);

		ConfigEntryString &operator=(const ConfigEntryString &entry);
		ConfigEntryString &operator=(const string &value);
		ConfigEntryString &operator=(const char *value);

		bool operator==(const ConfigEntryString &entry) const;
		bool operator!=(const ConfigEntryString &entry) const;
		bool operator==(const string &value) const;
		bool operator!=(const string &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryString &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryString &entry);
};


class ConfigEntryChoice : public ConfigEntry {
	protected:
		unsigned int _value;
		vector<string> value_name;
		vector<string> value_desc;

	public:
		ConfigEntryChoice();
		ConfigEntryChoice(const ConfigEntryChoice &entry);
		ConfigEntryChoice(string name, string desc);
		ConfigEntryChoice(string name, string desc, unsigned int value);
		ConfigEntryChoice(string name, string desc, const vector<string> &value_name);
		ConfigEntryChoice(string name, string desc, const char **value_name);
		ConfigEntryChoice(string name, string desc, const vector<string> &value_name, unsigned int value);
		ConfigEntryChoice(string name, string desc, const char **value_name, unsigned int value);

		const unsigned int &value() const;
		const unsigned int &operator()() const;
		void setValue(const unsigned int &value);
		void setValue(const string &value);

		void addChoice(unsigned int index, const string &name, const string &value_desc);
		void addChoice(const string &name, const string &value_desc);

		void setChoiceNames(const vector<string> &value_name);
		void setChoiceNames(const char **value_name);
		void setChoiceDescs(const vector<string> &value_desc);
		void setChoiceDescs(const char **value_desc);

		void choose(const char *value);
		void choose(const string &value);

		bool is(const string &value) const;

		ConfigEntryChoice &operator=(const ConfigEntryChoice &entry);
		ConfigEntryChoice &operator=(unsigned int value);
		ConfigEntryChoice &operator=(const string &value);
		ConfigEntryChoice &operator=(const char *value);

		bool operator==(const ConfigEntryChoice &entry) const;
		bool operator!=(const ConfigEntryChoice &entry) const;
		bool operator==(const unsigned int &value) const;
		bool operator!=(const unsigned int &value) const;
		bool operator==(const string &value) const;
		bool operator!=(const string &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryChoice &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryChoice &entry);
};


class ConfigEntryDouble : public ConfigEntry {
	protected:
		double _value;
		double _increment;
		bool _hasmin;
		double _min;
		bool _hasmax;
		double _max;

	public:
		ConfigEntryDouble();
		ConfigEntryDouble(const ConfigEntryDouble &entry);
		ConfigEntryDouble(string name, string desc);
		ConfigEntryDouble(string name, string desc, double value);
		ConfigEntryDouble(string name, string desc, double value, double increment);

		const double &value() const;
		const double &operator()() const;
		void setValue(const double &value);

		const double &increment() const;
		void setIncrement(const double &increment);

		const bool &hasmin() const;
		const double &min() const;
		void setMin(double min);
		void setNoMin();
		const bool &hasmax() const;
		const double &max() const;
		void setMax(double max);
		void setNoMax();

		ConfigEntryDouble &operator=(const ConfigEntryDouble &entry);
		ConfigEntryDouble &operator=(double value);

		bool operator==(const ConfigEntryDouble &entry) const;
		bool operator!=(const ConfigEntryDouble &entry) const;
		bool operator==(const double &value) const;
		bool operator!=(const double &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryDouble &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryDouble &entry);
};


class ConfigEntryUnsignedLong : public ConfigEntry {
	protected:
		unsigned long _value;
		unsigned long _increment;
		bool _hasmin;
		unsigned long _min;
		bool _hasmax;
		unsigned long _max;

	public:
		ConfigEntryUnsignedLong();
		ConfigEntryUnsignedLong(const ConfigEntryUnsignedLong &entry);
		ConfigEntryUnsignedLong(string name, string desc);
		ConfigEntryUnsignedLong(string name, string desc, unsigned long value);
		ConfigEntryUnsignedLong(string name, string desc, unsigned long value, unsigned long increment);

		const unsigned long &value() const;
		const unsigned long &operator()() const;
		void setValue(const unsigned long &value);

		const unsigned long &increment() const;
		void setIncrement(const unsigned long &increment);

		const bool &hasmin() const;
		const unsigned long &min() const;
		void setMin(unsigned long min);
		void setNoMin();
		const bool &hasmax() const;
		const unsigned long &max() const;
		void setMax(unsigned long max);
		void setNoMax();

		ConfigEntryUnsignedLong &operator=(const ConfigEntryUnsignedLong &entry);
		ConfigEntryUnsignedLong &operator=(unsigned long value);

		bool operator==(const ConfigEntryUnsignedLong &entry) const;
		bool operator!=(const ConfigEntryUnsignedLong &entry) const;
		bool operator==(const unsigned long &value) const;
		bool operator!=(const unsigned long &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryUnsignedLong &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryUnsignedLong &entry);
};


class ConfigEntryLong : public ConfigEntry {
	protected:
		long _value;
		long _increment;
		bool _hasmin;
		long _min;
		bool _hasmax;
		long _max;

	public:
		ConfigEntryLong();
		ConfigEntryLong(const ConfigEntryLong &entry);
		ConfigEntryLong(string name, string desc);
		ConfigEntryLong(string name, string desc, long value);
		ConfigEntryLong(string name, string desc, long value, long increment);

		const long &value() const;
		const long &operator()() const;
		void setValue(const long &value);

		const long &increment() const;
		void setIncrement(const long &increment);

		const bool &hasmin() const;
		const long &min() const;
		void setMin(long min);
		void setNoMin();
		const bool &hasmax() const;
		const long &max() const;
		void setMax(long max);
		void setNoMax();

		ConfigEntryLong &operator=(const ConfigEntryLong &entry);
		ConfigEntryLong &operator=(long value);

		bool operator==(const ConfigEntryLong &entry) const;
		bool operator!=(const ConfigEntryLong &entry) const;
		bool operator==(const long &value) const;
		bool operator!=(const long &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryLong &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryLong &entry);
};


class ConfigEntryBool : public ConfigEntry {
	protected:
		bool _value;

	public:
		ConfigEntryBool();
		ConfigEntryBool(const ConfigEntryBool &entry);
		ConfigEntryBool(string name, string desc);
		ConfigEntryBool(string name, string desc, bool value);

		const bool &value() const;
		const bool &operator()() const;
		operator bool() const;
		void setValue(const bool &value);

		ConfigEntryBool &operator=(const ConfigEntryBool &entry);
		ConfigEntryBool &operator=(bool value);

		bool operator==(const ConfigEntryBool &entry) const;
		bool operator!=(const ConfigEntryBool &entry) const;
		bool operator==(const bool &value) const;
		bool operator!=(const bool &value) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryBool &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryBool &entry);
};


class ConfigEntryTrigger : public ConfigEntry {
	protected:
		unsigned long _pending;

	public:
		ConfigEntryTrigger();
		ConfigEntryTrigger(const ConfigEntryTrigger &entry);
		ConfigEntryTrigger(string name, string desc);

		const unsigned long& pending() const;
		bool triggered() const;
		bool operator()() const;
		operator bool() const;
		void trigger();
		void untrigger();
		void setTrigger(bool trigger);

		ConfigEntryTrigger &operator=(const ConfigEntryTrigger &entry);
		ConfigEntryTrigger &operator=(bool triggered);

		bool operator==(const ConfigEntryTrigger &entry) const;
		bool operator!=(const ConfigEntryTrigger &entry) const;
		bool operator==(const bool &triggered) const;
		bool operator!=(const bool &triggered) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryTrigger &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryTrigger &entry);
};


class ConfigEntryDivider : public ConfigEntry {
	protected:

	public:
		ConfigEntryDivider();
		ConfigEntryDivider(const ConfigEntryDivider &entry);

		ConfigEntryDivider &operator=(const ConfigEntryDivider &entry);

		bool operator==(const ConfigEntryDivider &entry) const;
		bool operator!=(const ConfigEntryDivider &entry) const;

		virtual void output_definition_specific(ostream &out) const;
		virtual const char* get_type() const;
		virtual void output_value(ostream &out) const;

		virtual istream& input(istream& in);
		virtual ostream& output(ostream& out) const;
		friend istream& operator>>(istream& in, ConfigEntryDivider &entry);
		friend ostream& operator<<(ostream& out, const ConfigEntryDivider &entry);
};



#include "ConfigSet.hh"


#endif

