
// SimParm: Simple and flexible C++ configuration framework
// Copyright (C) 2007 Australian National University
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// Contact:
// Kevin Pulo
// kevin.pulo@anu.edu.au
// Leonard Huxley Bldg 56
// Australian National University, ACT, 0200, Australia

#include "ConfigEntry.hh"

ConfigEntry::ConfigEntry()
: _name(""),
  _desc(""),
  _invalid(false),
  _viewable(true),
  _editable(true),
  _outputOnChange(true),
  _configSet(NULL)
{
}

ConfigEntry::ConfigEntry(ConfigSet *configSet)
: _name(""),
  _desc(""),
  _invalid(false),
  _viewable(true),
  _editable(true),
  _outputOnChange(true),
  _configSet(configSet)
{
}

ConfigEntry::~ConfigEntry()
{
}

ConfigEntry::ConfigEntry(string name, string desc)
: _name(name),
  _desc(desc),
  _invalid(false),
  _viewable(true),
  _editable(true),
  _outputOnChange(true),
  _configSet(NULL)
{
}

ConfigEntry::ConfigEntry(string name, string desc, ConfigSet *configSet)
: _name(name),
  _desc(desc),
  _invalid(false),
  _viewable(true),
  _editable(true),
  _outputOnChange(true),
  _configSet(configSet)
{
}

ConfigEntry::ConfigEntry(const ConfigEntry &entry)
: _name(entry._name),
  _desc(entry._desc),
  _invalid(entry._invalid),
  _viewable(entry._viewable),
  _editable(entry._editable),
  _outputOnChange(entry._outputOnChange),
  _configSet(entry._configSet)
{
}

void ConfigEntry::setName(const string &name) {
	_name = name;
}

void ConfigEntry::setDesc(const string &desc) {
	_desc = desc;
}

void ConfigEntry::setInvalid(const bool &invalid) {
	_invalid = invalid;
}

void ConfigEntry::setViewable(const bool &viewable) {
	_viewable = viewable;
}

void ConfigEntry::setEditable(const bool &editable) {
	_editable = editable;
}

void ConfigEntry::setOutputOnChange(const bool &outputOnChange) {
	_outputOnChange = outputOnChange;
}

void ConfigEntry::setConfigSet(ConfigSet *configSet) {
	_configSet = configSet;
}

void ConfigEntry::addChangeCallback(changecallback_t callback) {
	changecallbacks.push_back(callback);
}

void ConfigEntry::removeChangeCallback(changecallback_t callback) {
	// FIXME
	//changecallbacks.erase(changecallbacks.find(callback));
}

const string& ConfigEntry::name() const {
	return _name;
}

const string& ConfigEntry::desc() const {
	return _desc;
}

const bool& ConfigEntry::invalid() const {
	return _invalid;
}

const bool& ConfigEntry::viewable() const {
	return _viewable;
}

const bool& ConfigEntry::editable() const {
	return _editable;
}

const bool& ConfigEntry::outputOnChange() const {
	return _outputOnChange;
}

const ConfigSet* ConfigEntry::configSet() const {
	return _configSet;
}


void ConfigEntry::notifyChangeCallbacks() const {
	for (vector<changecallback_t>::const_iterator callback = changecallbacks.begin(); callback != changecallbacks.end(); callback++) {
		(*callback)(this);
	}
	if (configSet()) {
		for (vector<changecallback_t>::const_iterator callback = configSet()->changecallbacks.begin(); callback != configSet()->changecallbacks.end(); callback++) {
			(*callback)(this);
		}
	}
}

void ConfigEntry::outputIfNecessary() const {
	if (outputOnChange()) {
		//cerr << "OUTPUT NECESSARY FOR " << name() << endl;
		output();
	}
	notifyChangeCallbacks();
}


ConfigEntry &ConfigEntry::operator=(const ConfigEntry &entry) {
	_name = entry._name;
	_desc = entry._desc;
	_invalid = entry._invalid;
	_viewable = entry._viewable;
	_editable = entry._editable;
	_outputOnChange = entry._outputOnChange;
	_configSet = entry._configSet;
	return *this;
}

void ConfigEntry::output_definition(ostream &out) const {
	out << "type " << get_type() << endl;
	out << "name " << name() << endl;
	out << "desc " << desc() << endl;
	out << "invalid " << boolalpha << invalid() << endl;
	out << "viewable " << boolalpha << viewable() << endl;
	out << "editable " << boolalpha << editable() << endl;
	out << "outputOnChange " << boolalpha << outputOnChange() << endl;
	output_definition_specific(out);
	out << "end" << endl;
}

void ConfigEntry::output_definition_specific(ostream &out) const {
}

const char* ConfigEntry::get_type() const {
	return "";
}

void ConfigEntry::output_value(ostream &out) const {
}

istream& operator>>(istream& in, ConfigEntry &entry) {
	return entry.input(in);
}

istream& ConfigEntry::input(istream& in) {
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntry &entry) {
	return entry.output(out);
}

void ConfigEntry::output() const {
	if (configSet() && configSet()->out && configSet()->attached) {
		ostream &o = *(configSet()->out);
		//cerr << "OUTPUTTING: " << name() << endl;
		o << "set ";
		output(o);
		//cerr << "FLUSHED" << endl;
		o.flush();  // THIS FLUSH SOMETIMES DOES NOT FLUSH.  Hence the need for the stupid heartbeat.
		fsync(configSet()->outfd);
		cout.flush();
		cerr.flush();
		fflush(stdout);
		fflush(stderr);
	} else {
		//cerr << "HUH? NO CONFIGSET!" << endl;
	}
}

ostream& ConfigEntry::output(ostream& out) const {
	out.flush();
	return out;
}


////////////////////////////////////////////////////
// STRING -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryString::ConfigEntryString()
: ConfigEntry(), _value("")
{
}

ConfigEntryString::ConfigEntryString(const ConfigEntryString &entry)
: ConfigEntry((ConfigEntry)*this), _value(entry._value)
{
}

ConfigEntryString::ConfigEntryString(string name, string desc)
: ConfigEntry(name, desc), _value("")
{
}

ConfigEntryString::ConfigEntryString(string name, string desc, const string &value)
: ConfigEntry(name, desc), _value(value)
{
}

ConfigEntryString::ConfigEntryString(string name, string desc, const char *value)
: ConfigEntry(name, desc), _value(string(value))
{
}

const string &ConfigEntryString::value() const {
	return _value;
}

const string &ConfigEntryString::operator()() const {
	return _value;
}

void ConfigEntryString::setValue(const string &value) {
	this->_value = value;
	outputIfNecessary();
}

ConfigEntryString &ConfigEntryString::operator=(const ConfigEntryString &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	return *this;
}

ConfigEntryString &ConfigEntryString::operator=(const string &value) {
	setValue(value);
	return *this;
}

ConfigEntryString &ConfigEntryString::operator=(const char *value) {
	setValue(string(value));
	return *this;
}

bool ConfigEntryString::operator==(const ConfigEntryString &entry) const {
	return (name() == entry.name()) && ( ! _value.compare(entry._value));
}

bool ConfigEntryString::operator!=(const ConfigEntryString &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryString::operator==(const string &value) const {
	return ( ! this->value().compare(value));
}

bool ConfigEntryString::operator!=(const string &value) const {
	return ! ( (*this) == value);
}

void ConfigEntryString::output_definition_specific(ostream &out) const {
	out << "value " << value() << endl;
}

const char* ConfigEntryString::get_type() const {
	return "string";
}

void ConfigEntryString::output_value(ostream &out) const {
	out << value();
}

istream& operator>>(istream& in, ConfigEntryString &entry) {
	return entry.input(in);
}

istream& ConfigEntryString::input(istream& in) {
	// must force it to read until \n, not just the next "word"
	const streamsize bufsize = 32*1024;
	char buf[bufsize];
	char ch;
	while (in.peek() == ' ' || in.peek() == '\t') {
		in.get(ch);
	}
	in.getline(buf, bufsize);
	if (in.bad()) {
		cerr << "Error: unable to read string value" << endl;
		setInvalid(true);
	} else {
		//_value = buf;
		setValue(buf);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryString &entry) {
	return entry.output(out);
}

ostream& ConfigEntryString::output(ostream& out) const {
	out << name() << " = " << value() << endl;
	out.flush();
	return out;
}



////////////////////////////////////////////////////
// CHOICE -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryChoice::ConfigEntryChoice()
: ConfigEntry()
{
}

ConfigEntryChoice::ConfigEntryChoice(const ConfigEntryChoice &entry)
: ConfigEntry((ConfigEntry)*this), _value(entry._value), value_name(entry.value_name)
{
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc)
: ConfigEntry(name, desc), _value(0)
{
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc, unsigned int value)
: ConfigEntry(name, desc), _value(value)
{
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc, const vector<string> &value_name)
: ConfigEntry(name, desc), _value(0), value_name(value_name)
{
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc, const char **value_name)
: ConfigEntry(name, desc), _value(0)
{
	setChoiceNames(value_name);
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc, const vector<string> &value_name, unsigned int value)
: ConfigEntry(name, desc), _value(value), value_name(value_name)
{
}

ConfigEntryChoice::ConfigEntryChoice(string name, string desc, const char **value_name, unsigned int value)
: ConfigEntry(name, desc), _value(value)
{
	setChoiceNames(value_name);
}

const unsigned int &ConfigEntryChoice::value() const {
	return _value;
}

const unsigned int &ConfigEntryChoice::operator()() const {
	return _value;
}

void ConfigEntryChoice::setValue(const unsigned int &value) {
	this->_value = value;
	outputIfNecessary();
}

void ConfigEntryChoice::setValue(const string &value) {
	choose(value);
}

void ConfigEntryChoice::addChoice(unsigned int index, const string &name, const string &value_desc) {
	if (index >= this->value_name.size()) {
		this->value_name.resize(index + 1);
		this->value_desc.resize(index + 1);
	}
	this->value_name[index] = name;
	this->value_desc[index] = value_desc;
}

void ConfigEntryChoice::addChoice(const string &name, const string &value_desc) {
	this->value_name.push_back(name);
	this->value_desc.push_back(value_desc);
}

void ConfigEntryChoice::setChoiceNames(const vector<string> &value_name) {
	this->value_name = value_name;
}

void ConfigEntryChoice::setChoiceNames(const char **value_name) {
	if (value_name) {
		for (unsigned int i = 0; value_name[i]; i++) {
			this->value_name.push_back(string(value_name[i]));
		}
	}
}

void ConfigEntryChoice::setChoiceDescs(const vector<string> &value_desc) {
	this->value_desc = value_desc;
}

void ConfigEntryChoice::setChoiceDescs(const char **value_desc) {
	if (value_desc) {
		for (unsigned int i = 0; value_desc[i]; i++) {
			this->value_desc.push_back(string(value_desc[i]));
		}
	}
}

void ConfigEntryChoice::choose(const char *value) {
	bool found = false;
	unsigned int index = 0;
	for (vector<string>::const_iterator it = value_name.begin(); it != value_name.end(); it++, index++) {
		if ( ! it->compare(value) ) {
			found = true;
			break;
		}
	}
	if (found) {
		setValue(index);
	} else {
		// check if the given value is an unsigned integer, if so, then atoui it
		// and use it as the value
		char *end = NULL;
		unsigned long v = strtoul(value, &end, 10);
		if (*value != '\0' && *end == '\0') {
			setValue((unsigned int)v);
		} else {
			cerr << "Warning: Attempt to set config " << name() << " to value " << value << " failed." << endl;
			//cerr << "Valid choices are..." << endl;
		}
	}
}

void ConfigEntryChoice::choose(const string &value) {
	choose(value.c_str());
}

bool ConfigEntryChoice::is(const string &value) const {
	return !value.compare(value_name[this->_value]);
}

ConfigEntryChoice &ConfigEntryChoice::operator=(const ConfigEntryChoice &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	this->value_name = entry.value_name;
	return *this;
}

ConfigEntryChoice &ConfigEntryChoice::operator=(unsigned int value) {
	setValue(value);
	return *this;
}

ConfigEntryChoice &ConfigEntryChoice::operator=(const string &value) {
	choose(value);
	return *this;
}

ConfigEntryChoice &ConfigEntryChoice::operator=(const char *value) {
	choose(value);
	return *this;
}

bool ConfigEntryChoice::operator==(const ConfigEntryChoice &entry) const {
	//return (name() == entry.name()) && (_value == entry._value) && (value_name == entry.value_name);
	return (name() == entry.name()) && (_value == entry._value);
}

bool ConfigEntryChoice::operator!=(const ConfigEntryChoice &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryChoice::operator==(const unsigned int &value) const {
	return (this->value() == value);
}

bool ConfigEntryChoice::operator!=(const unsigned int &value) const {
	return ! ( (*this) == value);
}

bool ConfigEntryChoice::operator==(const string &value) const {
	return is(value);
}

bool ConfigEntryChoice::operator!=(const string &value) const {
	return ! ( (*this) == value);
}

void ConfigEntryChoice::output_definition_specific(ostream &out) const {
	out << "value " << value() << endl;
	out << "num_values " << value_name.size() << endl;
	vector<string>::const_iterator n = value_name.begin();
	vector<string>::const_iterator desc = value_desc.begin();
	while (n != value_name.end() && desc != value_desc.end()) {
		out << *n << " " << *desc << endl;
		n++;
		desc++;
	}
}

const char* ConfigEntryChoice::get_type() const {
	return "choice";
}

void ConfigEntryChoice::output_value(ostream &out) const {
	out << value_name[value()];
}

istream& operator>>(istream& in, ConfigEntryChoice &entry) {
	return entry.input(in);
}

istream& ConfigEntryChoice::input(istream& in) {
	// must force it to read until \n, not just the next "word"
	const streamsize bufsize = 32*1024;
	char buf[bufsize];
	char ch;
	while (in.peek() == ' ' || in.peek() == '\t') {
		in.get(ch);
	}
	in.getline(buf, bufsize);
	if (in.bad()) {
		cerr << "Error: unable to read string value" << endl;
		setInvalid(true);
	} else {
		choose(buf);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryChoice &entry) {
	return entry.output(out);
}

ostream& ConfigEntryChoice::output(ostream& out) const {
	out << name() << " = " << value() << endl;
	//out << name() << " = " << value_name[value()] << endl;
	out.flush();
	return out;
}



////////////////////////////////////////////////////
// DOUBLE -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryDouble::ConfigEntryDouble()
: ConfigEntry(),
  _value(0.0),
  _increment(0.0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryDouble::ConfigEntryDouble(const ConfigEntryDouble &entry)
: ConfigEntry((ConfigEntry)*this),
  _value(entry._value),
  _increment(entry._increment),
  _hasmin(entry._hasmin),
  _min(entry._min),
  _hasmax(entry._hasmax),
  _max(entry._max)
{
}

ConfigEntryDouble::ConfigEntryDouble(string name, string desc)
: ConfigEntry(name, desc),
  _value(0.0),
  _increment(0.0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryDouble::ConfigEntryDouble(string name, string desc, double value)
: ConfigEntry(name, desc),
  _value(value),
  _increment(0.0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryDouble::ConfigEntryDouble(string name, string desc, double value, double increment)
: ConfigEntry(name, desc),
  _value(value),
  _increment(increment),
  _hasmin(false),
  _hasmax(false)
{
}

const double &ConfigEntryDouble::value() const {
	return _value;
}

const double &ConfigEntryDouble::operator()() const {
	return _value;
}

void ConfigEntryDouble::setValue(const double &value) {
	//cerr << "VALUE SET FOR " << name() << endl;
	this->_value = value;
	outputIfNecessary();
}

const double &ConfigEntryDouble::increment() const {
	return _increment;
}

void ConfigEntryDouble::setIncrement(const double &increment) {
	_increment = increment;
}

const bool &ConfigEntryDouble::hasmin() const {
	return _hasmin;
}

const double &ConfigEntryDouble::min() const {
	return _min;
}

void ConfigEntryDouble::setMin(double min) {
	_hasmin = true;
	_min = min;
}

void ConfigEntryDouble::setNoMin() {
	_hasmin = false;
}

const bool &ConfigEntryDouble::hasmax() const {
	return _hasmax;
}

const double &ConfigEntryDouble::max() const {
	return _max;
}

void ConfigEntryDouble::setMax(double max) {
	_hasmax = true;
	_max = max;
}

void ConfigEntryDouble::setNoMax() {
	_hasmax = false;
}

ConfigEntryDouble &ConfigEntryDouble::operator=(const ConfigEntryDouble &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	this->_increment = entry._increment;
	this->_hasmin = entry._hasmin;
	this->_min = entry._min;
	this->_hasmax = entry._hasmax;
	this->_max = entry._max;
	return *this;
}

ConfigEntryDouble &ConfigEntryDouble::operator=(double value) {
	//cerr << "ASSIGNMENT FOR " << name() << endl;
	setValue(value);
	return *this;
}

bool ConfigEntryDouble::operator==(const ConfigEntryDouble &entry) const {
	return (name() == entry.name()) && (_value == entry._value);
}

bool ConfigEntryDouble::operator!=(const ConfigEntryDouble &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryDouble::operator==(const double &value) const {
	return (this->value() == value);
}

bool ConfigEntryDouble::operator!=(const double &value) const {
	return ! ( (*this) == value );
}

void ConfigEntryDouble::output_definition_specific(ostream &out) const {
	out << "value " << value() << endl;
	out << "increment " << increment() << endl;
	out << "hasmin " << hasmin() << endl;
	if (hasmin()) {
		out << "min " << min() << endl;
	}
	out << "hasmax " << hasmax() << endl;
	if (hasmax()) {
		out << "max " << max() << endl;
	}
}

const char* ConfigEntryDouble::get_type() const {
	return "double";
}

void ConfigEntryDouble::output_value(ostream &out) const {
	out << value();
}

istream& operator>>(istream& in, ConfigEntryDouble &entry) {
	return entry.input(in);
}

istream& ConfigEntryDouble::input(istream& in) {
	double v;
	if ( ! (in >> v) ) {
		cerr << "Error: unable to read double value" << endl;
		setInvalid(true);
	} else {
		setValue(v);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryDouble &entry) {
	return entry.output(out);
}

ostream& ConfigEntryDouble::output(ostream& out) const {
	out << name() << " = " << value() << endl;
	out.flush();
	return out;
}




////////////////////////////////////////////////////
// UNSIGNED LONG -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryUnsignedLong::ConfigEntryUnsignedLong()
: ConfigEntry(),
  _value(0),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryUnsignedLong::ConfigEntryUnsignedLong(const ConfigEntryUnsignedLong &entry)
: ConfigEntry((ConfigEntry)*this),
  _value(entry._value),
  _increment(entry._increment),
  _hasmin(entry._hasmin),
  _min(entry._min),
  _hasmax(entry._hasmax),
  _max(entry._max)
{
}

ConfigEntryUnsignedLong::ConfigEntryUnsignedLong(string name, string desc)
: ConfigEntry(name, desc),
  _value(0),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryUnsignedLong::ConfigEntryUnsignedLong(string name, string desc, unsigned long value)
: ConfigEntry(name, desc),
  _value(value),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryUnsignedLong::ConfigEntryUnsignedLong(string name, string desc, unsigned long value, unsigned long increment)
: ConfigEntry(name, desc),
  _value(value),
  _increment(increment),
  _hasmin(false),
  _hasmax(false)
{
}

const unsigned long &ConfigEntryUnsignedLong::value() const {
	return _value;
}

const unsigned long &ConfigEntryUnsignedLong::operator()() const {
	return _value;
}

void ConfigEntryUnsignedLong::setValue(const unsigned long &value) {
	this->_value = value;
	outputIfNecessary();
}

const unsigned long &ConfigEntryUnsignedLong::increment() const {
	return _increment;
}

void ConfigEntryUnsignedLong::setIncrement(const unsigned long &increment) {
	_increment = increment;
}

const bool &ConfigEntryUnsignedLong::hasmin() const {
	return _hasmin;
}

const unsigned long &ConfigEntryUnsignedLong::min() const {
	return _min;
}

void ConfigEntryUnsignedLong::setMin(unsigned long min) {
	_hasmin = true;
	_min = min;
}

void ConfigEntryUnsignedLong::setNoMin() {
	_hasmin = false;
}

const bool &ConfigEntryUnsignedLong::hasmax() const {
	return _hasmax;
}

const unsigned long &ConfigEntryUnsignedLong::max() const {
	return _max;
}

void ConfigEntryUnsignedLong::setMax(unsigned long max) {
	_hasmax = true;
	_max = max;
}

void ConfigEntryUnsignedLong::setNoMax() {
	_hasmax = false;
}

ConfigEntryUnsignedLong &ConfigEntryUnsignedLong::operator=(const ConfigEntryUnsignedLong &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	this->_increment = entry._increment;
	this->_hasmin = entry._hasmin;
	this->_min = entry._min;
	this->_hasmax = entry._hasmax;
	this->_max = entry._max;
	return *this;
}

ConfigEntryUnsignedLong &ConfigEntryUnsignedLong::operator=(unsigned long value) {
	setValue(value);
	return *this;
}

bool ConfigEntryUnsignedLong::operator==(const ConfigEntryUnsignedLong &entry) const {
	return (name() == entry.name()) && (_value == entry._value);
}

bool ConfigEntryUnsignedLong::operator!=(const ConfigEntryUnsignedLong &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryUnsignedLong::operator==(const unsigned long &value) const {
	return (this->value() == value);
}

bool ConfigEntryUnsignedLong::operator!=(const unsigned long &value) const {
	return ! ( (*this) == value );
}

void ConfigEntryUnsignedLong::output_definition_specific(ostream &out) const {
	out << "value " << value() << endl;
	out << "increment " << increment() << endl;
	out << "hasmin " << hasmin() << endl;
	if (hasmin()) {
		out << "min " << min() << endl;
	}
	out << "hasmax " << hasmax() << endl;
	if (hasmax()) {
		out << "max " << max() << endl;
	}
}

const char* ConfigEntryUnsignedLong::get_type() const {
	return "unsigned_long";
}

void ConfigEntryUnsignedLong::output_value(ostream &out) const {
	out << value();
}

istream& operator>>(istream& in, ConfigEntryUnsignedLong &entry) {
	return entry.input(in);
}

istream& ConfigEntryUnsignedLong::input(istream& in) {
	unsigned long v;
	if ( ! (in >> v) ) {
		cerr << "Error: unable to read unsigned long value" << endl;
		setInvalid(true);
	} else {
		setValue(v);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryUnsignedLong &entry) {
	return entry.output(out);
}

ostream& ConfigEntryUnsignedLong::output(ostream& out) const {
	out << name() << " = " << value() << endl;
	out.flush();
	return out;
}



////////////////////////////////////////////////////
// LONG -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryLong::ConfigEntryLong()
: ConfigEntry(),
  _value(0),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryLong::ConfigEntryLong(const ConfigEntryLong &entry)
: ConfigEntry((ConfigEntry)*this),
  _value(entry._value),
  _increment(entry._increment),
  _hasmin(entry._hasmin),
  _min(entry._min),
  _hasmax(entry._hasmax),
  _max(entry._max)
{
}

ConfigEntryLong::ConfigEntryLong(string name, string desc)
: ConfigEntry(name, desc),
  _value(0),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryLong::ConfigEntryLong(string name, string desc, long value)
: ConfigEntry(name, desc),
  _value(value),
  _increment(0),
  _hasmin(false),
  _hasmax(false)
{
}

ConfigEntryLong::ConfigEntryLong(string name, string desc, long value, long increment)
: ConfigEntry(name, desc),
  _value(value),
  _increment(increment),
  _hasmin(false),
  _hasmax(false)
{
}

const long &ConfigEntryLong::value() const {
	return _value;
}

const long &ConfigEntryLong::operator()() const {
	return _value;
}

void ConfigEntryLong::setValue(const long &value) {
	this->_value = value;
	outputIfNecessary();
}

const long &ConfigEntryLong::increment() const {
	return _increment;
}

void ConfigEntryLong::setIncrement(const long &increment) {
	_increment = increment;
}

const bool &ConfigEntryLong::hasmin() const {
	return _hasmin;
}

const long &ConfigEntryLong::min() const {
	return _min;
}

void ConfigEntryLong::setMin(long min) {
	_hasmin = true;
	_min = min;
}

void ConfigEntryLong::setNoMin() {
	_hasmin = false;
}

const bool &ConfigEntryLong::hasmax() const {
	return _hasmax;
}

const long &ConfigEntryLong::max() const {
	return _max;
}

void ConfigEntryLong::setMax(long max) {
	_hasmax = true;
	_max = max;
}

void ConfigEntryLong::setNoMax() {
	_hasmax = false;
}

ConfigEntryLong &ConfigEntryLong::operator=(const ConfigEntryLong &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	this->_increment = entry._increment;
	this->_hasmin = entry._hasmin;
	this->_min = entry._min;
	this->_hasmax = entry._hasmax;
	this->_max = entry._max;
	return *this;
}

ConfigEntryLong &ConfigEntryLong::operator=(long value) {
	setValue(value);
	return *this;
}

bool ConfigEntryLong::operator==(const ConfigEntryLong &entry) const {
	return (name() == entry.name()) && (_value == entry._value);
}

bool ConfigEntryLong::operator!=(const ConfigEntryLong &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryLong::operator==(const long &value) const {
	return (this->value() == value);
}

bool ConfigEntryLong::operator!=(const long &value) const {
	return ! ( (*this) == value );
}

void ConfigEntryLong::output_definition_specific(ostream &out) const {
	out << "value " << value() << endl;
	out << "increment " << increment() << endl;
	out << "hasmin " << hasmin() << endl;
	if (hasmin()) {
		out << "min " << min() << endl;
	}
	out << "hasmax " << hasmax() << endl;
	if (hasmax()) {
		out << "max " << max() << endl;
	}
}

const char* ConfigEntryLong::get_type() const {
	return "long";
}

void ConfigEntryLong::output_value(ostream &out) const {
	out << value();
}

istream& operator>>(istream& in, ConfigEntryLong &entry) {
	return entry.input(in);
}

istream& ConfigEntryLong::input(istream& in) {
	long v;
	if ( ! (in >> v) ) {
		cerr << "Error: unable to read long value" << endl;
		setInvalid(true);
	} else {
		setValue(v);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryLong &entry) {
	return entry.output(out);
}

ostream& ConfigEntryLong::output(ostream& out) const {
	out << name() << " = " << value() << endl;
	out.flush();
	return out;
}




////////////////////////////////////////////////////
// BOOL -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryBool::ConfigEntryBool()
: ConfigEntry(), _value(false)
{
}

ConfigEntryBool::ConfigEntryBool(const ConfigEntryBool &entry)
: ConfigEntry((ConfigEntry)*this), _value(entry._value)
{
}

ConfigEntryBool::ConfigEntryBool(string name, string desc)
: ConfigEntry(name, desc), _value(false)
{
}

ConfigEntryBool::ConfigEntryBool(string name, string desc, bool value)
: ConfigEntry(name, desc), _value(value)
{
}

const bool &ConfigEntryBool::value() const {
	return _value;
}

const bool &ConfigEntryBool::operator()() const {
	return _value;
}

ConfigEntryBool::operator bool() const {
	return _value;
}

void ConfigEntryBool::setValue(const bool &value) {
	//cerr << "VALUE SET FOR " << name() << endl;
	this->_value = value;
	outputIfNecessary();
}

ConfigEntryBool &ConfigEntryBool::operator=(const ConfigEntryBool &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_value = entry._value;
	return *this;
}

ConfigEntryBool &ConfigEntryBool::operator=(bool value) {
	//cerr << "ASSIGNMENT FOR " << name() << endl;
	setValue(value);
	return *this;
}

bool ConfigEntryBool::operator==(const ConfigEntryBool &entry) const {
	return (name() == entry.name()) && (_value == entry._value);
}

bool ConfigEntryBool::operator!=(const ConfigEntryBool &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryBool::operator==(const bool &value) const {
	return (this->value() == value);
}

bool ConfigEntryBool::operator!=(const bool &value) const {
	return ! ( (*this) == value );
}

void ConfigEntryBool::output_definition_specific(ostream &out) const {
	out << "value " << boolalpha << value() << endl;
}

const char* ConfigEntryBool::get_type() const {
	return "bool";
}

void ConfigEntryBool::output_value(ostream &out) const {
	out << value();
}

istream& operator>>(istream& in, ConfigEntryBool &entry) {
	return entry.input(in);
}

istream& ConfigEntryBool::input(istream& in) {
	bool v;
	if ( ! (in >> boolalpha >> v) ) {
		cerr << "Error: unable to read bool value" << endl;
		setInvalid(true);
	} else {
		setValue(v);
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryBool &entry) {
	return entry.output(out);
}

ostream& ConfigEntryBool::output(ostream& out) const {
	out << name() << " = " << boolalpha << value() << endl;
	out.flush();
	return out;
}





////////////////////////////////////////////////////
// TRIGGER -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryTrigger::ConfigEntryTrigger()
: ConfigEntry(), _pending(0)
{
}

ConfigEntryTrigger::ConfigEntryTrigger(const ConfigEntryTrigger &entry)
: ConfigEntry((ConfigEntry)*this), _pending(entry._pending)
{
}

ConfigEntryTrigger::ConfigEntryTrigger(string name, string desc)
: ConfigEntry(name, desc), _pending(0)
{
}

const unsigned long &ConfigEntryTrigger::pending() const {
	return _pending;
}

bool ConfigEntryTrigger::triggered() const {
	return (pending() > 0);
}

bool ConfigEntryTrigger::operator()() const {
	return triggered();
}

ConfigEntryTrigger::operator bool() const {
	return triggered();
}

void ConfigEntryTrigger::trigger() {
	_pending++;
	outputIfNecessary();
}

void ConfigEntryTrigger::untrigger() {
	if (_pending > 0) {
		_pending--;
		outputIfNecessary();
	}
}

void ConfigEntryTrigger::setTrigger(bool trigger) {
	if (trigger) {
		this->trigger();
	} else {
		this->untrigger();
	}
}

ConfigEntryTrigger &ConfigEntryTrigger::operator=(const ConfigEntryTrigger &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	this->_pending = entry._pending;
	return *this;
}

ConfigEntryTrigger &ConfigEntryTrigger::operator=(bool triggered) {
	setTrigger(triggered);
	return *this;
}

bool ConfigEntryTrigger::operator==(const ConfigEntryTrigger &entry) const {
	return (name() == entry.name()) && (triggered() == entry.triggered());
}

bool ConfigEntryTrigger::operator!=(const ConfigEntryTrigger &entry) const {
	return ! ( (*this) == entry );
}

bool ConfigEntryTrigger::operator==(const bool &triggered) const {
	return (this->triggered() == triggered);
}

bool ConfigEntryTrigger::operator!=(const bool &triggered) const {
	return ! ( (*this) == triggered );
}

void ConfigEntryTrigger::output_definition_specific(ostream &out) const {
	out << "pending " << pending() << endl;
}

const char* ConfigEntryTrigger::get_type() const {
	return "trigger";
}

void ConfigEntryTrigger::output_value(ostream &out) const {
	out << pending();
}

istream& operator>>(istream& in, ConfigEntryTrigger &entry) {
	return entry.input(in);
}

istream& ConfigEntryTrigger::input(istream& in) {
	string s;
	if ( ! (in >> s) ) {
		cerr << "Error: unable to read trigger value" << endl;
		setInvalid(true);
	}
	if ( ! s.compare("trigger") ) {
		trigger();
	} else {
		cerr << "Error: trigger value can only be \"trigger\"" << endl;
	}
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryTrigger &entry) {
	return entry.output(out);
}

ostream& ConfigEntryTrigger::output(ostream& out) const {
	out << name() << " = " << pending() << endl;
	out.flush();
	return out;
}






////////////////////////////////////////////////////
// DIVIDER -------------------------------------------
////////////////////////////////////////////////////


ConfigEntryDivider::ConfigEntryDivider()
: ConfigEntry()
{
}

ConfigEntryDivider::ConfigEntryDivider(const ConfigEntryDivider &entry)
: ConfigEntry((ConfigEntry)*this)
{
}

ConfigEntryDivider &ConfigEntryDivider::operator=(const ConfigEntryDivider &entry) {
	(ConfigEntry)(*this) = (ConfigEntry)entry;
	return *this;
}

bool ConfigEntryDivider::operator==(const ConfigEntryDivider &entry) const {
	return true;
}

bool ConfigEntryDivider::operator!=(const ConfigEntryDivider &entry) const {
	return ! ( (*this) == entry );
}

void ConfigEntryDivider::output_definition_specific(ostream &out) const {
}

const char* ConfigEntryDivider::get_type() const {
	return "divider";
}

void ConfigEntryDivider::output_value(ostream &out) const {
}

istream& operator>>(istream& in, ConfigEntryDivider &entry) {
	return entry.input(in);
}

istream& ConfigEntryDivider::input(istream& in) {
	return in;
}

ostream& operator<<(ostream& out, const ConfigEntryDivider &entry) {
	return entry.output(out);
}

ostream& ConfigEntryDivider::output(ostream& out) const {
	// hope this doesn't break anything in the twiddler, etc...
	// it "shouldn't" break...
	out << "--------------------------------" << endl;
	out.flush();
	return out;
}

